/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.chuck;

import org.apache.camel.Component;
import org.apache.camel.Consumer;
import org.apache.camel.Processor;
import org.apache.camel.Producer;
import org.apache.camel.component.chuck.service.ApiService;
import org.apache.camel.component.chuck.service.ApiServiceImpl;
import org.apache.camel.spi.UriEndpoint;
import org.apache.camel.spi.UriParam;
import org.apache.camel.support.DefaultEndpoint;
import org.apache.camel.util.ObjectHelper;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.AsyncHttpClientConfig;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClientConfig;
import org.asynchttpclient.proxy.ProxyServer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@UriEndpoint(firstVersion = "3.0.0", scheme = "chuck", title = "Chuck", syntax = "chuck:type", label = "chuck", producerOnly = true)
public class ChuckEndpoint extends DefaultEndpoint {

    private static final Logger LOG = LoggerFactory.getLogger(ChuckEndpoint.class);

    @UriParam
    private ChuckConfiguration configuration;

    private AsyncHttpClient client;
    private AsyncHttpClientConfig clientConfig;
    private ApiService service;

    public ChuckEndpoint(
            String endpointUri,
            Component component,
            ChuckConfiguration configuration,
            AsyncHttpClient client,
            AsyncHttpClientConfig clientConfig) {
        super(endpointUri, component);
        this.configuration = configuration;
        this.client = client;
        this.clientConfig = clientConfig;
    }

    @Override
    protected void doStart() throws Exception {
        super.doStart();
        // create one HTTP client instance to be used across out the application 
        if (client == null) {
            DefaultAsyncHttpClientConfig.Builder builder = clientConfig != null
                    ? new DefaultAsyncHttpClientConfig.Builder(clientConfig)
                    : new DefaultAsyncHttpClientConfig.Builder();

            if (configuration != null && ObjectHelper.isNotEmpty(configuration.getProxyHost())
                    && ObjectHelper.isNotEmpty(configuration.getProxyPort())) {
                LOG.debug("Setup http proxy host:{} port:{} for ChuckService", configuration.getProxyHost(),
                        configuration.getProxyPort());
                builder.setProxyServer(
                        new ProxyServer.Builder(configuration.getProxyHost(), configuration.getProxyPort()).build());
            }
            final AsyncHttpClientConfig config = builder.build();
            client = new DefaultAsyncHttpClient(config);
        }
        if (service == null) {
            service = new ApiServiceImpl(
                    client,
                    configuration.getBufferSize(),
                    configuration.getBaseUrl());
        }
    }

    @Override
    protected void doStop() throws Exception {
        super.doStop();
        // close the HTTP client to prevent any memory leak
        if (client != null && !client.isClosed()) {
            client.close();
        }
        client = null;
    }

    @Override
    public Producer createProducer() throws Exception {
        return new ChuckProducer(this);
    }

    @Override
    public Consumer createConsumer(Processor processor) throws Exception {
        // in case of a polling API use ScheduledPollEndpoint and ScheduledPollConsumer
        throw new UnsupportedOperationException("No support for consumers");
    }

    public ChuckConfiguration getConfiguration() {
        return configuration;
    }

    public void setConfiguration(ChuckConfiguration configuration) {
        this.configuration = configuration;
    }

    public ApiService getApiService() {
        return service;
    }

}
