/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.chuck;

import java.util.Map;

import org.apache.camel.Endpoint;
import org.apache.camel.spi.Metadata;
import org.apache.camel.spi.annotations.Component;
import org.apache.camel.support.DefaultComponent;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.AsyncHttpClientConfig;

@Component("chuck")
public class ChuckComponent extends DefaultComponent {

    @Metadata(label = "advanced", defaultValue = ChuckConfiguration.API_BASE_URL, 
        description = "Can be used to set an alternative base URI, e.g. when you want to test the component against a mock API.")
    private String baseUrl = ChuckConfiguration.API_BASE_URL;
    private AsyncHttpClient client;
    @Metadata(label = "advanced", description = "Configure the AsyncHttpClient to use a custom com.ning.http.client.AsyncHttpClientConfig instance.")
    private AsyncHttpClientConfig clientConfig;
    
    public ChuckComponent() {
    }

    @Override
    protected Endpoint createEndpoint(final String uri, String remaining, final Map<String, Object> parameters)
            throws Exception {
        final ChuckConfiguration configuration = new ChuckConfiguration();

        // ignore trailing slash
        if (remaining.endsWith("/")) {
            remaining = remaining.substring(0, remaining.length() - 1);
        }
        configuration.setType(remaining);

        if (!remaining.equals(ChuckConfiguration.JOKES_TYPE)) {
            throw new IllegalArgumentException("Unsupported endpoint type for uri " + uri + ", remaining " + remaining);
        }
        if (configuration.getBaseUrl() == null) {
            configuration.setBaseUrl(baseUrl);
        }

        final ChuckEndpoint endpoint = new ChuckEndpoint(uri, this, configuration, client, clientConfig);
        setProperties(endpoint, parameters);

        return endpoint;
    }

    public void setBaseUrl(final String baseUrl) {
        this.baseUrl = baseUrl;
    }

    public String getBaseUrl() {
        return baseUrl;
    }

    public AsyncHttpClient getClient() {
        return client;
    }

    public AsyncHttpClientConfig getClientConfig() {
        return clientConfig;
    }

    public void setClientConfig(final AsyncHttpClientConfig clientConfig) {
        this.clientConfig = clientConfig;
    }

}
