/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.chuck;

import java.net.HttpURLConnection;

import com.github.tomakehurst.wiremock.WireMockServer;

import org.apache.camel.CamelContext;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.chuck.model.RandomJoke;
import org.apache.camel.component.mock.MockEndpoint;
import org.apache.camel.test.AvailablePortFinder;
import org.apache.camel.test.junit5.CamelTestSupport;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;

import static com.github.tomakehurst.wiremock.client.WireMock.*;
import static org.junit.Assert.assertEquals;

public class ChuckComponentTest extends CamelTestSupport {

    private static volatile int port;

    private static WireMockServer wireMockServer;
 
    @BeforeAll
    public static void startServer() {
        port = AvailablePortFinder.getNextAvailable();
        wireMockServer = new WireMockServer(port);
        wireMockServer.start();
        setupStubs();
    }
 
    @AfterAll
    public static void stopServer() {
        wireMockServer.stop();
        wireMockServer = null;
    }
 
    public static void setupStubs() {
        wireMockServer.stubFor(get(urlEqualTo("/jokes/random"))
            .willReturn(aResponse()
                .withHeader("Content-Type", "application/json; charset=UTF-8")
                .withStatus(HttpURLConnection.HTTP_OK)
                .withBodyFile("json/random-joke.json")));
    }

    @Override
    protected CamelContext createCamelContext() throws Exception {
        final CamelContext context = super.createCamelContext();
        context.disableJMX();
        final ChuckComponent component = new ChuckComponent();
        component.setBaseUrl("http://localhost:" + port);
        /*AsyncHttpClientConfig clientConfig = 
            new DefaultAsyncHttpClientConfig.Builder().setConnectTimeout(1000).build();
        component.setClientConfig(clientConfig);*/
        context.addComponent("chuck", component);
        return context;
    }

    @Override
    protected int getShutdownTimeout() {
        return 60;
    }

    @Test
    public void testRandomJoke() throws Exception {
        MockEndpoint mock = getMockEndpoint("mock:result");
        mock.expectedMinimumMessageCount(1);
        mock.expectedBodyReceived().body(RandomJoke.class);
        final RandomJoke response = (RandomJoke) template.requestBody("direct:test", "");
        assertEquals("GlLdJ3LLSKapU-tK92V8Rw", response.getId());
        assertMockEndpointsSatisfied();
    }

    @Override
    protected RouteBuilder createRouteBuilder() throws Exception {
        return new RouteBuilder() {
            public void configure() {
                from("direct:test")
                    .to("chuck:jokes")
                    .log("${body}")
                    .to("mock:result");
            }
        };
    }

}
